defmodule Jeu.Huit do
  @moduledoc """
  Implémentation du jeu de 8 américain.
  """

  @behaviour Regles

  defstruct actuel: 0,
            défausse: Pile.new(),
            mains: %{},
            noms: %{},
            pioche: Pile.new(),
            sens: :ordinaire,
            visible: nil

  @typedoc "Un jeu de 8 américain"
  @type t() :: %{
          actuel: integer(),
          défausse: Pile.t(),
          mains: %{integer() => Pile.t()},
          noms: %{integer() => String.t()},
          pioche: Pile.t(),
          sens: :ordinaire | :inverse,
          visible: Carte.t()
        }

  @impl true
  @doc """
  Retourne le titre du jeu sous la forme d'une chaîne de caractères.
  """
  @spec titre() :: String.t()
  def titre, do: "8 américain"

  @impl true
  @doc """
  Crée une nouvelle structure.

  Cette fonction crée la structure propre au jeu du 8 américain et prépare
  la partie.

  ## Exemples

      iex> jeu = Jeu.Huit.new
      iex> Pile.taille(jeu.pioche)
      51
      iex> jeu.visible == nil
      false

  """
  @spec new() :: struct()
  def new do
    pioche = Pile.new(52) |> Pile.mélanger()
    {visibles, pioche} = Pile.retirer(pioche, 1)
    %Jeu.Huit{pioche: pioche, visible: Enum.at(visibles, 0)}
  end

  @impl true
  @doc """
  Retourne `true` si on peut ajouter un nouveau joueur au jeu, `false` sinon.

  On ne peut ajouter plus de 5 joueurs (7 * 5 = 35 cartes).

  ## Exemples

      iex> jeu = Jeu.Huit.new
      iex> Jeu.Huit.ajouter_joueur?(jeu)
      true

      iex> jeu = Jeu.Huit.new
      iex> noms = for i <- 1..5, into: %{}, do: {i, ""}
      iex> jeu = %{jeu | noms: noms}
      iex> Jeu.Huit.ajouter_joueur?(jeu)
      false

  """
  @spec ajouter_joueur?(struct()) :: boolean()
  def ajouter_joueur?(jeu) do
    map_size(jeu.noms) < 5
  end

  @impl true
  @doc """
  Ajoute un joueur.

  On doit préciser en paramètres le jeu lui-même et le nom du joueur
  à ajouter. `ajouter_joueur?` doit avoir été appelée auparavant.
  Un tuple est retourné contenant le jeu modifié et l'identifiant
  du joueur ajouté (un entier).

  ## Exemples

      iex> jeu = Jeu.Huit.new
      iex> map_size(jeu.mains)
      0

      iex> {jeu, id} = Jeu.Huit.new |> Jeu.Huit.ajouter_joueur("Vincent")
      iex> id
      0
      iex> map_size(jeu.mains)
      1
      iex> map_size(jeu.noms)
      1
      iex> Pile.taille(jeu.mains[0])
      7
      iex> Pile.taille(jeu.pioche)
      44

  """
  @spec ajouter_joueur(struct(), String.t()) :: struct()
  def ajouter_joueur(jeu, nom) do
    {main, pioche} = Pile.retirer(jeu.pioche, 7)
    identifiant = map_size(jeu.mains)
    mains = Map.put(jeu.mains, identifiant, main)
    noms = Map.put(jeu.noms, identifiant, nom)
    jeu = %{jeu | mains: mains, noms: noms, pioche: pioche}
    {jeu, identifiant}
  end

  @impl true
  @doc """
  Est-ce que ce coup est permis ?

  Retourne soit `true` soit `false`.
  On doit préciser en paramètres le jeu en lui-même, le joueur sous forme
  d'entier et le coup à jouer : ce dernier peut être un atome (comme
  `:piocher`) ou un tuple (comme `{:jouer, %Carte{}}`).

  ## Exemples

  Un joueur dont ce n'est pas le tour de jouer retourne toujours `false` :

      iex> {jeu, _} = Jeu.Huit.new
      ...> |> Jeu.Huit.ajouter_joueur("Vincent")
      iex> Jeu.Huit.jouer?(jeu, 1, :piocher)
      false

  Un joueur ne peut pas jouer une carte qu'il ne possède pas :

      iex> jeu = %Jeu.Huit{visible: Carte.new("5", "carreau")}
      iex> {jeu, _} = Jeu.Huit.ajouter_joueur(jeu, "Vincent")
      iex> mains = %{
      ...>   0 => Pile.new
      ...> }
      iex> jeu = %{jeu | mains: mains}
      iex> Jeu.Huit.jouer?(jeu, 0, {:jouer, "10", "trèfle"})
      false
      iex> Jeu.Huit.jouer?(jeu, 0, :piocher)
      true

  Un joueur possédant une carte de même valeur peut jouer mais ne peut
  pas piocher :

      iex> jeu = %Jeu.Huit{visible: Carte.new("5", "carreau")}
      iex> {jeu, _} = Jeu.Huit.ajouter_joueur(jeu, "Vincent")
      iex> mains = %{
      ...>   0 => Pile.new |> Pile.ajouter("5", "trèfle")
      ...> }
      iex> jeu = %{jeu | mains: mains}
      iex> Jeu.Huit.jouer?(jeu, 0, {:jouer, "5", "trèfle"})
      true
      iex> Jeu.Huit.jouer?(jeu, 0, :piocher)
      false

  De même si le joueur a une carte de même enseigne :

      iex> jeu = %Jeu.Huit{visible: Carte.new("5", "carreau")}
      iex> {jeu, _} = Jeu.Huit.ajouter_joueur(jeu, "Vincent")
      iex> mains = %{
      ...>   0 => Pile.new |> Pile.ajouter("10", "carreau")
      ...> }
      iex> jeu = %{jeu | mains: mains}
      iex> Jeu.Huit.jouer?(jeu, 0, {:jouer, "10", "carreau"})
      true
      iex> Jeu.Huit.jouer?(jeu, 0, :piocher)
      false

  Si le joueur ne peut pas jouer, il peut piocher :

      iex> jeu = %Jeu.Huit{visible: Carte.new("5", "carreau")}
      iex> {jeu, _} = Jeu.Huit.ajouter_joueur(jeu, "Vincent")
      iex> mains = %{
      ...>   0 => Pile.new |> Pile.ajouter("10", "trèfle")
      ...> }
      iex> jeu = %{jeu | mains: mains}
      iex> Jeu.Huit.jouer?(jeu, 0, {:jouer, "10", "trèfle"})
      false
      iex> Jeu.Huit.jouer?(jeu, 0, :piocher)
      true

  Un 8 est toujours accepté cependant :

      iex> jeu = %Jeu.Huit{visible: Carte.new("5", "carreau")}
      iex> {jeu, _} = Jeu.Huit.ajouter_joueur(jeu, "Vincent")
      iex> mains = %{
      ...>   0 => Pile.new |> Pile.ajouter("8", "trèfle")
      ...> }
      iex> jeu = %{jeu | mains: mains}
      iex> Jeu.Huit.jouer?(jeu, 0, {:jouer, "8", "trèfle"})
      true
      iex> Jeu.Huit.jouer?(jeu, 0, :piocher)
      false

  """
  @spec jouer?(struct(), integer(), Regles.coup()) :: boolean()
  def jouer?(jeu, joueur, _) when joueur != jeu.actuel, do: false

  def jouer?(jeu, joueur, :piocher) do
    Enum.all?(jeu.mains[joueur], fn carte -> not jouer?(jeu, joueur, {:jouer, carte}) end)
  end

  def jouer?(jeu, joueur, {:jouer, valeur, enseigne}) do
    carte = Carte.new(valeur, enseigne)

    case carte do
      %Carte{} -> jouer?(jeu, joueur, {:jouer, carte})
      _ -> false
    end
  end

  def jouer?(jeu, joueur, {:jouer, %Carte{valeur: "8"} = carte}) do
    carte in jeu.mains[joueur]
  end

  def jouer?(jeu, joueur, {:jouer, %Carte{} = carte}) do
    carte in jeu.mains[joueur] and
      (carte.valeur == jeu.visible.valeur or carte.enseigne == jeu.visible.enseigne)
  end

  def jouer?(_jeu, _joueur, _coup), do: false

  @impl true
  @doc """
  Fait jouer un joueur.

  On doit spécifier en paramètres le jeu lui-même, le joueur sous
  la forme d'un entier et le coup à réaliser (un atome ou un tuple dont
  le premier élément est un atome).
  La fonction `jouer?` doit avoir été appelée auparavant.
  Le jeu modifié est retourné.

  ## Exemples

  Si le joueur joue un 2, le joueur suivant doit piocher deux cartes et
  passe son tour :

      iex> pioche =
      ...>   Pile.new
      ...>   |> Pile.ajouter("3", "pique")
      ...>   |> Pile.ajouter("valet", "carreau")
      iex> mains = %{
      ...>   0 => Pile.new |> Pile.ajouter("2", "trèfle"),
      ...>   1 => Pile.new,
      ...>   2 => Pile.new
      ...> }
      iex> visible = Carte.new("7", "trèfle")
      iex> jeu = %Jeu.Huit{mains: mains, pioche: pioche, visible: visible}
      iex> jeu = Jeu.Huit.jouer(jeu, 0, {:jouer, "2", "trèfle"})
      iex> {"2", "trèfle"} in jeu.mains[0]
      false
      iex> {"2", "trèfle"} in jeu.défausse
      true
      iex> jeu.actuel
      2
      iex> Pile.taille(jeu.mains[1])
      2
      iex> jeu.visible
      %Carte{enseigne: "trèfle", valeur: "2"}

  Si le joueur joue un valet, le joueur suivant passe son tour :

      iex> pioche = Pile.new
      iex> mains = %{
      ...>   0 => Pile.new |> Pile.ajouter("valet", "carreau"),
      ...>   1 => Pile.new,
      ...>   2 => Pile.new
      ...> }
      iex> visible = Carte.new("7", "carreau")
      iex> jeu = %Jeu.Huit{mains: mains, pioche: pioche, visible: visible}
      iex> jeu = Jeu.Huit.jouer(jeu, 0, {:jouer, "valet", "carreau"})
      iex> {"valet", "carreau"} in jeu.mains[0]
      false
      iex> {"valet", "carreau"} in jeu.défausse
      true
      iex> jeu.actuel
      2
      iex> Pile.taille(jeu.mains[1])
      0

  Un as change de sens :

      iex> pioche = Pile.new
      iex> mains = %{
      ...>   0 => Pile.new |> Pile.ajouter("as", "pique"),
      ...>   1 => Pile.new,
      ...>   2 => Pile.new |> Pile.ajouter("as", "carreau")
      ...> }
      iex> visible = Carte.new("7", "pique")
      iex> jeu = %Jeu.Huit{mains: mains, pioche: pioche, visible: visible}
      iex> jeu = Jeu.Huit.jouer(jeu, 0, {:jouer, "as", "pique"})
      iex> {"as", "pique"} in jeu.mains[0]
      false
      iex> {"as", "pique"} in jeu.défausse
      true
      iex> jeu.actuel
      2
      iex> jeu.sens
      :inverse
      iex> jeu = Jeu.Huit.jouer(jeu, 2, {:jouer, "as", "carreau"})
      iex> jeu.actuel
      0
      iex> jeu.sens
      :ordinaire

  Une autre carte n'a aucun effet :

      iex> pioche = Pile.new
      iex> mains = %{
      ...>   0 => Pile.new |> Pile.ajouter("10", "carreau"),
      ...>   1 => Pile.new,
      ...>   2 => Pile.new
      ...> }
      iex> visible = Carte.new("7", "carreau")
      iex> jeu = %Jeu.Huit{mains: mains, pioche: pioche, visible: visible}
      iex> jeu = Jeu.Huit.jouer(jeu, 0, {:jouer, "10", "carreau"})
      iex> {"10", "carreau"} in jeu.mains[0]
      false
      iex> {"10", "carreau"} in jeu.défausse
      true
      iex> jeu.visible
      %Carte{enseigne: "carreau", valeur: "10"}
      iex> jeu.actuel
      1

  """
  @spec jouer(struct(), integer(), Regles.coup()) :: struct()
  def jouer(jeu, _joueur, :piocher) do
    jeu
    |> piocher(jeu.actuel)
    |> valider_tour
  end

  def jouer(jeu, joueur, {:jouer, valeur, enseigne}) do
    carte = Carte.new(valeur, enseigne)

    case carte do
      %Carte{} -> jouer(jeu, joueur, {:jouer, carte})
      _ -> :invalide
    end
  end

  def jouer(jeu, _joueur, {:jouer, %Carte{valeur: "1"} = carte}) do
    jeu
    |> jouer_carte(carte)
    |> changer_sens
    |> valider_tour
  end

  def jouer(jeu, joueur, {:jouer, %Carte{valeur: "2"} = carte}) do
    jeu
    |> jouer_carte(carte)
    |> piocher(joueur_suivant(jeu, joueur), 2)
    |> valider_tour(2)
  end

  def jouer(jeu, _joueur, {:jouer, %Carte{valeur: "valet"} = carte}) do
    jeu
    |> jouer_carte(carte)
    |> valider_tour(2)
  end

  def jouer(jeu, _joueur, {:jouer, %Carte{} = carte}) do
    jeu
    |> jouer_carte(carte)
    |> valider_tour
  end

  @spec jouer_carte(t(), Carte.t()) :: t()
  defp jouer_carte(jeu, carte) do
    main = jeu.mains[jeu.actuel]
    {main, défausse} = Pile.transférer(main, jeu.défausse, carte)
    mains = Map.put(jeu.mains, jeu.actuel, main)
    %{jeu | défausse: défausse, mains: mains, visible: carte}
  end

  @spec valider_tour(t(), integer()) :: t()
  defp valider_tour(jeu, nombre \\ 1) do
    actuel = Enum.reduce(1..nombre, jeu.actuel, fn _, pos -> joueur_suivant(jeu, pos) end)
    %{jeu | actuel: actuel}
  end

  @spec joueur_suivant(t(), integer()) :: integer()
  defp joueur_suivant(jeu, actuel) do
    maximum = map_size(jeu.mains) - 1

    case jeu.sens do
      :ordinaire -> (actuel >= maximum && 0) || actuel + 1
      :inverse -> (actuel > 0 && actuel - 1) || maximum
    end
  end

  @spec changer_sens(t()) :: t()
  defp changer_sens(jeu) do
    sens =
      case jeu.sens do
        :ordinaire -> :inverse
        :inverse -> :ordinaire
      end

    %{jeu | sens: sens}
  end

  @spec piocher(t(), integer(), integer()) :: t()
  defp piocher(jeu, joueur, nombre \\ 1) do
    {pioche, défausse} =
      if Pile.taille(jeu.pioche) < nombre do
        pioche = Pile.fusionner(jeu.défausse, jeu.pioche) |> Pile.mélanger()
        {pioche, Pile.new()}
      else
        {jeu.pioche, jeu.défausse}
      end

    {piochées, pioche} = Pile.retirer(pioche, nombre)
    main = Pile.fusionner(jeu.mains[joueur], piochées)
    mains = Map.put(jeu.mains, joueur, main)
    %{jeu | défausse: défausse, mains: mains, pioche: pioche}
  end
end
